
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:25:55 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version DX_CRYPTO_DES.c#1:csrc:6
   *  \author adams
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */



/************* Include Files ****************/

/* .............. CRYS level includes ................. */
#include <crypto/algapi.h>
#include <linux/list.h>
#include <linux/module.h>
#include <linux/crypto.h>

#include "BaseTypes.h"
#include "error.h"
#include "CRYS_DES_error.h"
#include "CRYS_DES.h"

/************************ Defines ******************************/
#define DES_BLOCK_SIZE     8
#define DES3_128_BLOCK_SIZE (2*DES_BLOCK_SIZE)
#define DES3_192_BLOCK_SIZE (3*DES_BLOCK_SIZE)

#define DES_MIN_KEY_SIZE   8
#define AES_MAX_KEY_SIZE   24

#define DES_KEY_SIZE        8
#define DES3_128_KEY_SIZE (2*DES_KEY_SIZE)
#define DES3_192_KEY_SIZE (3*DES_KEY_SIZE)

#define CRYPT_S390_PRIORITY 300
#define CRYPT_S390_COMPOSITE_PRIORITY 400
/************************ Global Data ******************************/


/************* Private function prototype ****************/

/************************ Public Functions ******************************/
int dx_des_init(void);
void dx_des_finish(void);

static int dx_des_setkey(struct crypto_tfm *tfm, const uint8_t *in_key,uint key_len);
static int dx_ecb_des_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					                            struct scatterlist *src, uint nbytes);
static int dx_ecb_des_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					                            struct scatterlist *src, uint nbytes);
static int dx_cbc_des_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					                            struct scatterlist *src, uint nbytes);
static int dx_cbc_des_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					                            struct scatterlist *src, uint nbytes);

static int dx_des3_128_setkey(struct crypto_tfm *tfm, const uint8_t *in_key, uint key_len);
static int dx_ecb_des3_128_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						                            struct scatterlist *src, uint nbytes);
static int dx_ecb_des3_128_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						                            struct scatterlist *src, uint nbytes);
static int dx_cbc_des3_128_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						                            struct scatterlist *src, uint nbytes);
static int dx_cbc_des3_128_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						                            struct scatterlist *src, uint nbytes);

static int dx_des3_192_setkey(struct crypto_tfm *tfm, const uint8_t *in_key, uint key_len);
static int dx_ecb_des3_192_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						                                struct scatterlist *src, uint nbytes);
static int dx_ecb_des3_192_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						                                struct scatterlist *src, uint nbytes);
static int dx_cbc_des3_192_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						                                struct scatterlist *src, uint nbytes);
static int dx_cbc_des3_192_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						                                struct scatterlist *src, uint nbytes);

/************************ Structures ******************************/
/* Global structure for discretix des context */
struct dx_des_ctx {
	uint8_t iv[DES_BLOCK_SIZE];
	uint8_t key[DES_KEY_SIZE];
};
  
struct dx_des3_128_ctx {
         uint8_t iv[DES_BLOCK_SIZE];
         uint8_t key[DES3_128_KEY_SIZE];
  };
  
struct dx_des3_192_ctx {
	uint8_t iv[DES_BLOCK_SIZE];
	uint8_t key[DES3_192_KEY_SIZE];
};

/************* DES  1 KEY - ECB structure ***************/
static struct crypto_alg ecb_des_alg = {
	.cra_name        = "ecb(des)",
        .cra_driver_name = "ecb-des-s390",
        .cra_priority    = CRYPT_S390_COMPOSITE_PRIORITY,
        .cra_flags       = CRYPTO_ALG_TYPE_BLKCIPHER,
        .cra_blocksize   = DES_BLOCK_SIZE,
        .cra_ctxsize     = sizeof(struct dx_des_ctx),
        .cra_type        = &crypto_blkcipher_type,
        .cra_module      = THIS_MODULE,
        .cra_list        = LIST_HEAD_INIT(ecb_des_alg.cra_list),
        .cra_u           = {
		.blkcipher = {
			.min_keysize = DES_KEY_SIZE,
			.max_keysize = DES_KEY_SIZE,
			.setkey      = dx_des_setkey,
			.encrypt     = dx_ecb_des_encrypt,
			.decrypt     = dx_ecb_des_decrypt,
		}
	}
};

/************* DES  1 KEY - CBC structure ***************/ 
 static struct crypto_alg cbc_des_alg = {
	.cra_name        = "cbc(des)",
        .cra_driver_name = "cbc-des-s390",
        .cra_priority    = CRYPT_S390_COMPOSITE_PRIORITY,
        .cra_flags       = CRYPTO_ALG_TYPE_BLKCIPHER,
        .cra_blocksize   = DES_BLOCK_SIZE,
        .cra_ctxsize     = sizeof(struct dx_des_ctx),
        .cra_type        = &crypto_blkcipher_type,
        .cra_module      = THIS_MODULE,
        .cra_list        = LIST_HEAD_INIT(cbc_des_alg.cra_list),
        .cra_u           = {
		.blkcipher = {
			.min_keysize = DES_KEY_SIZE,
			.max_keysize = DES_KEY_SIZE,
			.ivsize      = DES_BLOCK_SIZE,
			.setkey      = dx_des_setkey,
			.encrypt     = dx_cbc_des_encrypt,
			.decrypt     = dx_cbc_des_decrypt,
		}
	}
};

/************* DES  2 KEYS - ECB structure ***************/
static struct crypto_alg ecb_des3_128_alg = {
	.cra_name        = "ecb(des3_ede128)",
	.cra_driver_name = "ecb-des3_ede128-s390",
	.cra_priority    = CRYPT_S390_COMPOSITE_PRIORITY,
	.cra_flags       = CRYPTO_ALG_TYPE_BLKCIPHER,
	.cra_blocksize   = DES3_128_BLOCK_SIZE,
	.cra_ctxsize     = sizeof(struct dx_des3_128_ctx),
	.cra_type        = &crypto_blkcipher_type,
	.cra_module      = THIS_MODULE,
	.cra_list        = LIST_HEAD_INIT(ecb_des3_128_alg.cra_list),
	.cra_u           = {
		.blkcipher = {
			.min_keysize = DES3_128_KEY_SIZE,
			.max_keysize = DES3_128_KEY_SIZE,
			.setkey      = dx_des3_128_setkey,
			.encrypt     = dx_ecb_des3_128_encrypt,
			.decrypt     = dx_ecb_des3_128_decrypt,
		}
	}
 };

 /************* DES  2 KEYS - CBC structure ***************/
 static struct crypto_alg cbc_des3_128_alg = {
	.cra_name        = "cbc(des3_ede128)",
	.cra_driver_name = "cbc-des3_ede128-s390",
	.cra_priority    = CRYPT_S390_COMPOSITE_PRIORITY,
	.cra_flags       = CRYPTO_ALG_TYPE_BLKCIPHER,
	.cra_blocksize   = DES3_128_BLOCK_SIZE,
	.cra_ctxsize     = sizeof(struct dx_des3_128_ctx),
	.cra_type        = &crypto_blkcipher_type,
	.cra_module      = THIS_MODULE,
	.cra_list        = LIST_HEAD_INIT(cbc_des3_128_alg.cra_list),
	.cra_u           = {
		.blkcipher = {
			.min_keysize = DES3_128_KEY_SIZE,
			.max_keysize = DES3_128_KEY_SIZE,
			.ivsize      = DES3_128_BLOCK_SIZE,
			.setkey      = dx_des3_128_setkey,
			.encrypt     = dx_cbc_des3_128_encrypt,
			.decrypt     = dx_cbc_des3_128_decrypt,
		}
	}
};

/************* DES 3 KEYS - ECB *structure **************/
 static struct crypto_alg ecb_des3_192_alg = {
	.cra_name        = "ecb(des3_ede)",
	.cra_driver_name = "ecb-des3_ede-s390",
	.cra_priority    = CRYPT_S390_COMPOSITE_PRIORITY,
	.cra_flags       = CRYPTO_ALG_TYPE_BLKCIPHER,
	.cra_blocksize   = DES3_192_BLOCK_SIZE,
	.cra_ctxsize     = sizeof(struct dx_des3_192_ctx),
	.cra_type        = &crypto_blkcipher_type,
	.cra_module      = THIS_MODULE,
	.cra_list        = LIST_HEAD_INIT(ecb_des3_192_alg.cra_list),
	.cra_u           = {
		.blkcipher = {
			.min_keysize = DES3_192_KEY_SIZE,
			.max_keysize = DES3_192_KEY_SIZE,
			.setkey      = dx_des3_192_setkey,
			.encrypt     = dx_ecb_des3_192_encrypt,
			.decrypt     = dx_ecb_des3_192_decrypt,
		}
	}
};

/************* DES 3 KEYS - CBC structure ***************/
 static struct crypto_alg cbc_des3_192_alg = {
	.cra_name        = "cbc(des3_ede)",
	.cra_driver_name = "cbc-des3_ede-s390",
	.cra_priority    = CRYPT_S390_COMPOSITE_PRIORITY,
	.cra_flags       = CRYPTO_ALG_TYPE_BLKCIPHER,
	.cra_blocksize   = DES3_192_BLOCK_SIZE,
	.cra_ctxsize     = sizeof(struct dx_des3_192_ctx),
	.cra_type        = &crypto_blkcipher_type,
	.cra_module      = THIS_MODULE,
	.cra_list        = LIST_HEAD_INIT(cbc_des3_192_alg.cra_list),
	.cra_u           = {
		.blkcipher = {
			.min_keysize = DES3_192_KEY_SIZE,
			.max_keysize = DES3_192_KEY_SIZE,
			.ivsize      = DES3_192_BLOCK_SIZE,
			.setkey      = dx_des3_192_setkey,
			.encrypt     = dx_cbc_des3_192_encrypt,
			.decrypt     = dx_cbc_des3_192_decrypt,
		}
	}
};

/****************************************************************************************************/
/**
 * @brief This function is used to registreate the des algorithm.
 *			 It initilize the crypto_alg structure for ecb and cbc,
 *			 and call the registration function
 *   
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* des.h
 */

int dx_des_init(void)
{

/* The return error identifier */
uint32_t error;
	
/* initializing the error to O.K */
error = DX_OK; 
 
	/* call the registration function with DES 1 KEYS - ECB struct */
	error = crypto_register_alg(&ecb_des_alg);
	if (error!=DX_OK)
		goto ecb_des_err;
         
        /* call the registration function with DES 1 KEYS - CBC struct */
	error = crypto_register_alg(&cbc_des_alg);
	if (error!=DX_OK)
		goto cbc_des_err;
         
        /* call the registration function with DES 2 KEYS - ECB struct */
	error = crypto_register_alg(&ecb_des3_128_alg);
	if (error!=DX_OK)
		goto ecb_des3_128_err;
         
        /* call the registration function with DES 2 KEYS - CBC struct */
	error = crypto_register_alg(&cbc_des3_128_alg);
	if (error!=DX_OK)
		goto cbc_des3_128_err;
         
        /* call the registration function with DES 3 KEYS - ECB struct */
	error = crypto_register_alg(&ecb_des3_192_alg);
	if (error!=DX_OK)
		goto ecb_des3_192_err;
         
        /* call the registration function with DES 3 KEYS - CBC struct */
	error = crypto_register_alg(&cbc_des3_192_alg);
	if (error!=DX_OK)
		goto cbc_des3_192_err;

	return DX_OK;
	
cbc_des3_192_err:
	crypto_unregister_alg(&ecb_des3_192_alg);
ecb_des3_192_err:
	crypto_unregister_alg(&cbc_des3_128_alg);
cbc_des3_128_err:
	crypto_unregister_alg(&ecb_des3_128_alg);
ecb_des3_128_err:         
	crypto_unregister_alg(&cbc_des_alg);
cbc_des_err:
	crypto_unregister_alg(&ecb_des_alg);
ecb_des_err:
	return error;
 }


/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *   
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* des.h
 */
void dx_des_finish(void)
{
	crypto_unregister_alg(&cbc_des3_192_alg);
	crypto_unregister_alg(&ecb_des3_192_alg);
        crypto_unregister_alg(&cbc_des3_128_alg);
        crypto_unregister_alg(&ecb_des3_128_alg);
        crypto_unregister_alg(&cbc_des_alg);
        crypto_unregister_alg(&ecb_des_alg);
 }
 
/****************************************************************************************************
 ****************************************** DES 1 KEY ***********************************************
 ****************************************************************************************************/
 
/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 * 
 * @param[in] tfm - pointer to the high level structure crypto_tfm.
 * @param[in] in_key - A pointer to the user's key buffer.
 * @param[in] key_len - The size of the KEY used by the DES: 8, 16 or 24 bytes.
 *								The size in bytes.
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_des_setkey(struct crypto_tfm *tfm, const uint8_t *key, unsigned int keylen)
{
/* The return error identifier */
uint32_t error;
   
struct dx_des_ctx *ctx_ptr = crypto_tfm_ctx(tfm);
        
/* initializing the error to O.K */
error = DX_OK; 
 
	if (keylen !=DES_KEY_SIZE)
		return CRYS_DES_ILLEGAL_NUM_OF_KEYS_ERROR;
    
	memcpy(ctx_ptr->key, key, keylen);
      
return DX_OK;
	
};


/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *   
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_ecb_des_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					struct scatterlist *src, unsigned int nbytes)
{
/* The return error identifier */
uint32_t error;
  
struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
   
	/* Locking data in memory */

	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_1_KeyInUse,
			CRYS_DES_Encrypt, CRYS_DES_ECB_mode, (uint8_t*)src, nbytes,
									(uint8_t*)dst);
	if (error != DX_OK)
		return error;
	
	/* releasing the out buffer in the memory */
	return DX_OK;	
};

/**
 * @brief This function is used to unregistreate the DES algorithm.
 *
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_ecb_des_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
                                	struct scatterlist *src, unsigned int nbytes)
{

/* The return error identifier */
uint32_t error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
        
	/* Locking data in memory */

	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_1_KeyInUse,
			CRYS_DES_Decrypt, CRYS_DES_ECB_mode, (uint8_t*)src, nbytes,         
									(uint8_t*)dst);
	if (error != DX_OK)
		return error;

	return DX_OK;	  	
};


/**
 * @brief This function is used to unregistreate the DES algorithm.
 *   
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_cbc_des_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					struct scatterlist *src, unsigned int nbytes)
{
/* The return error identifier */
uint32_t error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
        
	/* Locking data in memory */

	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_1_KeyInUse,
			CRYS_DES_Encrypt, CRYS_DES_CBC_mode, (uint8_t*)src, nbytes,         
								     (uint8_t*)dst);
	if (error != DX_OK)
		return error;
	
	/* releasing the out buffer in the memory */
	return DX_OK;	
};

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_cbc_des_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					struct scatterlist *src, unsigned int nbytes)
{

/* The return error identifier */
uint32_t error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
        
	/* Locking data in memory */

	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_1_KeyInUse,
			CRYS_DES_Decrypt, CRYS_DES_CBC_mode, (uint8_t*)src, nbytes,         
									(uint8_t*)dst);
	if (error != DX_OK)
		return error;
	
	return DX_OK;	  	
};

/****************************************************************************************************
 ****************************************** DES 2 KEYS **********************************************
 ****************************************************************************************************/

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 * 
 * @param[in] tfm - pointer to the high level structure crypto_tfm.
 * @param[in] in_key - A pointer to the user's key buffer.
 * @param[in] key_len - The size of the KEY used by the DES: 8, 16 or 24 bytes.
 *								The size in bytes.
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_des3_128_setkey(struct crypto_tfm *tfm, const uint8_t *key, unsigned int keylen)
{
/* The return error identifier */
uint32_t error;
   
struct dx_des_ctx *ctx_ptr = crypto_tfm_ctx(tfm);
        
/* initializing the error to O.K */
error = DX_OK; 
 
	if (keylen !=DES3_128_KEY_SIZE)
		return CRYS_DES_ILLEGAL_NUM_OF_KEYS_ERROR;
    
	memcpy(ctx_ptr->key, key, keylen);
    
	return DX_OK;
};

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *   
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_ecb_des3_128_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
						struct scatterlist *src, unsigned int nbytes)
{
/* The return error identifier */
uint32_t error;
  
struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
   
	/* Locking data in memory */
	
	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_2_KeysInUse,
			CRYS_DES_Encrypt, CRYS_DES_ECB_mode, (uint8_t*)src, nbytes,
								    (uint8_t*)dst);
	if (error != DX_OK)
		return error;
	  
	/* releasing the out buffer in the memory */
	return DX_OK;	
};

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_ecb_des3_128_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					    struct scatterlist *src, unsigned int nbytes)
{

/* The return error identifier */
uint32_t error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
        
	/* Locking data in memory */

   	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_2_KeysInUse,
			CRYS_DES_Decrypt, CRYS_DES_ECB_mode, (uint8_t*)src, nbytes,         
								     (uint8_t*)dst);
	if (error != DX_OK)
		return error;
	
	return DX_OK;	  	
};

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *   
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_cbc_des3_128_encrypt( struct blkcipher_desc *desc, struct scatterlist *dst,
						struct scatterlist *src, unsigned int nbytes)
{
/* The return error identifier */
uint32_t error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
        
	/* Locking data in memory */
   	
	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_2_KeysInUse,
			CRYS_DES_Encrypt, CRYS_DES_CBC_mode, (uint8_t*)src, nbytes,         
								    (uint8_t*)dst);
	if (error != DX_OK)
		return error;
	  
	/* releasing the out buffer in the memory */
	
	return DX_OK;	
};
/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_cbc_des3_128_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					    struct scatterlist *src, unsigned int nbytes)
{

/* The return error identifier */
uint32_t            error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
        
	/* Locking data in memory */

	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_2_KeysInUse,
			CRYS_DES_Decrypt, CRYS_DES_CBC_mode, (uint8_t*)src, nbytes,         
								     (uint8_t*)dst);
	if (error != DX_OK)
		return error;
	
	return DX_OK;	  	
};

/****************************************************************************************************
 ****************************************** DES 3 KEYS **********************************************
 ****************************************************************************************************/

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 * 
 * @param[in] tfm - pointer to the high level structure crypto_tfm.
 * @param[in] in_key - A pointer to the user's key buffer.
 * @param[in] key_len - The size of the KEY used by the DES: 8, 16 or 24 bytes.
 *								The size in bytes.
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_des3_192_setkey(struct crypto_tfm *tfm, const uint8_t *key, unsigned int keylen)
{
/* The return error identifier */
uint32_t            error;
   
struct dx_des_ctx *ctx_ptr = crypto_tfm_ctx(tfm);
        
/* initializing the error to O.K */
error = DX_OK; 

	if (keylen !=DES3_192_KEY_SIZE)
		return CRYS_DES_ILLEGAL_NUM_OF_KEYS_ERROR;
    
      	memcpy(ctx_ptr->key, key, keylen);

	return DX_OK;

};

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *   
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_ecb_des3_192_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					    struct scatterlist *src, unsigned int nbytes)
{
/* The return error identifier */
uint32_t            error;
  
struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
   
	/* Locking data in memory */
	
	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_3_KeysInUse,
			CRYS_DES_Encrypt, CRYS_DES_ECB_mode, (uint8_t*)src, nbytes,         
								    (uint8_t*)dst);
	if (error != DX_OK)
		return error;
	  
	/* releasing the out buffer in the memory */
	return DX_OK;	
};
/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_ecb_des3_192_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					    struct scatterlist *src, unsigned int nbytes)
{

/* The return error identifier */
uint32_t error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
        
	/* Locking data in memory */
   	
	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_3_KeysInUse,
				CRYS_DES_Decrypt, CRYS_DES_ECB_mode, (uint8_t*)src,
								nbytes, (uint8_t*)dst);
	if (error != DX_OK)
		return error;
	
	return DX_OK;	  	
};

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *   
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_cbc_des3_192_encrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
                                    	    struct scatterlist *src, unsigned int nbytes)
{
/* The return error identifier */
uint32_t error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 
        
	/* Locking data in memory */
   	
	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_3_KeysInUse,
                        	CRYS_DES_Encrypt, CRYS_DES_CBC_mode, (uint8_t*)src,
								nbytes, (uint8_t*)dst);
	if (error != DX_OK)
		return error;
	
	/* releasing the out buffer in the memory */
	return DX_OK;	
};

/****************************************************************************************************/
/**
 * @brief This function is used to unregistreate the DES algorithm.
 *
 * @param[in] desc - pointer to the high level structure blkcipher_desc.
 * @param[in] Dst - pointer to the destination structure scatterlist.
 * @param[in] Src - pointer to the source structure scatterlist.
 * @param[in] nbytes - number of bytes to encrypt
 *
 * @return value - On success DX_OK is returned, on failure a
 *                        value MODULE_* CRYS_DES_error.h
 */
static int dx_cbc_des3_192_decrypt(struct blkcipher_desc *desc, struct scatterlist *dst,
					    struct scatterlist *src, unsigned int nbytes)
{

/* The return error identifier */
uint32_t error;

struct dx_des_ctx *ctx_ptr = crypto_blkcipher_ctx(desc->tfm);
  
/* initializing the error to O.K */
error = DX_OK; 

	/* Locking data in memory */

	/* call the DES function with the rlevant parameters for encrypt ecb mode */
	error = CRYS_DES(DX_NULL, (CRYS_DES_Key_t*)ctx_ptr->key, CRYS_DES_3_KeysInUse,
				CRYS_DES_Decrypt, CRYS_DES_CBC_mode, (uint8_t*)src,        
								nbytes, (uint8_t*)dst);
	if (error != DX_OK)
		return error;

	return DX_OK;	  	
};


/**************************** Dummy function ********************/
 uint32_t  CRYS_DES(CRYS_DES_Iv_t            IV_ptr,
                                 CRYS_DES_Key_t*          Key_ptr,
                                 CRYS_DES_NumOfKeys_t     NumOfKeys,
                                 CRYS_DES_EncryptMode_t   EncryptDecryptFlag,
                                 CRYS_DES_OperationMode_t OperationMode,             
                                 uint8_t*               DataIn_ptr,             /* in */ 
                                 uint32_t               DataInSize,              /* in */ 
                                 uint8_t*               DataOut_ptr ) 
{
    printk(KERN_ALERT "\n HELLO: CRYS_DES - DUMMY FUNCTION\n"); 
    return DX_OK;
}
